/*
 * @(#)AbstractSimulation.java  2.2  2006-02-28
 *
 * Copyright (c) 2004-2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.simulation;

import ch.randelshofer.util.*;
import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import javax.swing.event.*;
import java.util.*;
import javax.swing.undo.*;
import java.util.concurrent.*;
/**
 * AbstractSimulation.
 * 
 * 
 * @author Werner Randelshofer
 * @version 2.2 2006-02-28 Write SimulatedConnections after SimulatedObject's.
 * <br>2.1 2006-02-21 Element handler added.
 * <br>2.0 2006-02-07 Reworked for Java 1.5.
 * <br>1.0 5. Februar 2004  Created.
 */
public abstract class AbstractSimulation implements Simulation {
    protected EventListenerList listenerList = new EventListenerList();
    private HashSet<SimulatedObject> objects = new HashSet<SimulatedObject>();
    private Executor behaviourDispatcher = Executors.newSingleThreadExecutor();
    private SimulatedObjectListener elementHandler = new SimulatedObjectListener() {
        public void relationshipAdded(SimulatedObjectEvent e) {
        }
        
        public void relationshipRemoved(SimulatedObjectEvent e) {
        }
        
        public void objectChanged(SimulatedObjectEvent e) {
        }
        
        public void objectAdded(SimulatedObjectEvent e) {
        }
        
        public void objectRemoved(SimulatedObjectEvent e) {
        }
        
        public void objectRequestRemove(SimulatedObjectEvent e) {
            remove(e.getObject());
        }
        
    };
    
    
    /** Creates a new instance. */
    public AbstractSimulation() {
    }
    
    public void add(SimulatedObject element) {
        if (! objects.contains(element)) {
            objects.add(element);
            element.addNotify(this);
            element.addSimulatedObjectListener(elementHandler);
            fireObjectAdded(element);
        }
    }
    
    public Collection<SimulatedObject> getObjects() {
        return (objects == null) ? Collections.EMPTY_SET : Collections.unmodifiableSet(objects);
    }
    
    public Collection<SimulatedElement> getElements() {
        LinkedList<SimulatedElement> list = new LinkedList<SimulatedElement>();
        for (SimulatedObject o: objects) {
            if (o instanceof SimulatedElement) {
                list.add((SimulatedElement) o);
            }
        }
        return list;
    }
    public Collection<SimulatedRelationship> getRelationships() {
        LinkedList<SimulatedRelationship> list = new LinkedList<SimulatedRelationship>();
        for (SimulatedObject o: objects) {
            if (o instanceof SimulatedRelationship) {
                list.add((SimulatedRelationship) o);
            }
        }
        return list;
    }
    
    public void remove(SimulatedObject element) {
        if (objects.contains(element)) {
            element.removeSimulatedObjectListener(elementHandler);
            element.removeNotify(this);
            objects.remove(element);
            fireObjectRemoved(element);
        }
    }
    
    public int getObjectCount() {
        return objects.size();
    }
    public int getElementCount() {
        int count = 0;
        for (SimulatedObject e : objects) {
            if ( !(e instanceof SimulatedRelationship)) {
                count++;
            }
        }
        return count;
    }
    public int getConnectionCount() {
        int count = 0;
        for (SimulatedObject e : objects) {
            if (e instanceof SimulatedRelationship) {
                count++;
            }
        }
        return count;
    }
    
    /**
     * Adds a listener for SimulationEvent's.
     */
    public void addSimulationListener(SimulationListener l) {
        listenerList.add(SimulationListener.class, l);
    }
    
    /**
     * Removes a listener for ElementEvent's.
     */
    public void removeSimulationListener(SimulationListener l) {
        listenerList.remove(SimulationListener.class, l);
    }
    
    
    public void addUndoableEditListener(javax.swing.event.UndoableEditListener l) {
        listenerList.add(UndoableEditListener.class, l);
    }
    
    public void removeUndoableEditListener(javax.swing.event.UndoableEditListener l) {
        listenerList.remove(UndoableEditListener.class, l);
    }
    
    public void fireUndoableEditHappened(javax.swing.undo.UndoableEdit edit) {
        if (listenerList != null) {
            UndoableEditEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==UndoableEditListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new UndoableEditEvent(this, edit);
                    ((UndoableEditListener)listeners[i+1]).undoableEditHappened(event);
                }
            }
        }
    }
    
    
    
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.  The event instance
     * is lazily created using the parameters passed into
     * the fire method.
     */
    protected void fireObjectAdded(SimulatedObject element) {
        if (listenerList != null) {
            SimulationEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==SimulationListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new SimulationEvent(this, element);
                    ((SimulationListener)listeners[i+1]).objectAdded(event);
                }
            }
        }
    }
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.  The event instance
     * is lazily created using the parameters passed into
     * the fire method.
     */
    protected void fireObjectRemoved(SimulatedObject element) {
        if (listenerList != null) {
            SimulationEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==SimulationListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new SimulationEvent(this, element);
                    ((SimulationListener)listeners[i+1]).objectRemoved(event);
                }
            }
        }
    }
    
    public void read(DOMInput in) throws IOException {
        for (int i=0; i < in.getElementCount(); i++) {
            add((SimulatedObject) in.readObject(i));
        }
    }
    
    public void write(DOMOutput out) throws IOException {
        // Write SimulatedConnections after other SimulatedObject's.
        // This makes the XML document easier to read.
        for (SimulatedObject elem : objects) {            
            if (! (elem instanceof SimulatedRelationship)) {
                out.writeObject(elem);
            }
        }
        for (SimulatedObject elem : objects) {            
            if ((elem instanceof SimulatedRelationship)) {
                out.writeObject(elem);
            }
        }
    }
    
    /**
     * Invokes the run method of the provided Runnable object on
     * the behaviour thread of this simulation.
     */
    public void invokeBehaviour(final Runnable runner) {
        Runnable r = new Runnable() {
            public void run() {
                runner.run();
                    /*
                fireBehaviourStarted();
                try {
                    runner.run();
                } finally {
                    fireBehaviourFinished();
                }*/
            }
        };
        behaviourDispatcher.execute(r);
    }
    
    public long getDelay() {
        return 1000;
    }
}
